;======================================================================
; ESMF_regrid_36.ncl
;
; Concepts illustrated:
;   - Use NCL coordinate subscripting to extract a region from a global grid
;   - Interpolating a regional rectilinear variable to a WRF grid
;   - Perform 4 different interpolation methods
;======================================================================
;No need to load these after 6.1.2
;load "$NCARG_ROOT/lib/ncarg/nclscripts/csm/gsn_code.ncl"
;load "$NCARG_ROOT/lib/ncarg/nclscripts/csm/gsn_csm.ncl"
;load "$NCARG_ROOT/lib/ncarg/nclscripts/csm/contributed.ncl"

; must load the 'ESMF-regridding.ncl' library
load "$NCARG_ROOT/lib/ncarg/nclscripts/esmf/ESMF_regridding.ncl"

begin
;----------------------------------------------------------------------
; Read source and destination grids, and variable to be regridded
;----------------------------------------------------------------------

;---Source grid is rectilinear (EDGAR)
  path_edgar = "./"
  fn_co2     = "era5_pressure_1983_01_01_00.nc";era5_z500_1980_01.nc" ;v42_CO2_2008_TOT.0.1x0.1.nc"
  f_co2      = addfile(path_edgar+fn_co2,"r")
  sfile      = f_co2 
  emi_co2    = f_co2->Z_GDS0_ISBL ;Z500(0,:,:);emi_co2
  printVarSummary(emi_co2)

;---Destination grid is curvilinear (WRF)
  path_wrf   = "./";/lus/theta-fs0/projects/Climate_Water/NRCM_DD/WRF_CCSM_BC/" ;lat_lon_coarsen.nc/./"
  fwrf       = addfile(path_wrf+"lat_lon_coarsen.nc","r")
  lat2d      = fwrf->XLAT(0,:,:)
  lon2d      = fwrf->XLONG(0,:,:)

;---Create an 'extra' large boundary around the WRF grid
  extra      = 2          ; arbitrary
  minLat2D   = min(lat2d) - extra
  maxLat2D   = max(lat2d) + extra
  minLon2D   = min(lon2d) - extra
  maxLon2D   = max(lon2d) + extra

;----------------------------------------------------------------------
; Extract *only* the EDGAR grid area surrounding China
; Use NCL's coordinate subscripting which use the {...} syntax
; This grid subset is now regional and not global.
;----------------------------------------------------------------------

  var = emi_co2({minLat2D:maxLat2D},{minLon2D:maxLon2D})

;----------------------------------------------------------------------
; It is always useful to 'look-at-the-data' be used. Print useful information.
;----------------------------------------------------------------------
  printVarSummary(var)    ; original units
  printMinMax(var,0)
  print("==================================================")
  print("Source grid")
  print("   Dimensions       : " + str_join(""+dimsizes(var)," x "))
;  print("   Min/max lat      : " + min(var&lat) + "/" + max(var&lat))
;  print("   Min/max lon      : " + min(var&lon) + "/" + max(var&lon))
  print("   # valid values   : " + num(.not.ismissing(var)))
  print("   # missing values : " + num(ismissing(var)))
  print("==================================================")
  print("Destination grid")
  print("   Dimensions       : " + str_join(""+dimsizes(lat2d)," x "))
  print("   Min/max lat      : " + min(lat2d) + "/" + max(lat2d))
  print("   Min/max lon      : " + min(lon2d) + "/" + max(lon2d))
  print("==================================================")

;----------------------------------------------------------------------
; Change units (not required)
;----------------------------------------------------------------------

  conv       = 1. ;e9*3600/44  ; EDGAR in kg m-2 s-1  WRFchem need   mol km^-2 hr^-1 
  var        = (/var*conv /) 
  var@units  = "m";ol/km^2 hr^1"

  printVarSummary(var)
  printMinMax(var,0)
  print("==================================================")

;----------------------------------------------------------------------
; Statistical distribution of t he source variable
; Use this as a guide to set plot limits.
; Note: There are outliers
;----------------------------------------------------------------------

  opt = True
  opt@PrintStat = True
  stat_var = stat_dispersion(var, opt )

; snip
; [3]          LowDec=0
; [4]          LowOct=0
; [5]          LowSex=0
; [6]     LowQuartile=4.05303
; [7]          LowTri=13.7226
; [8]          Median=59.8534
; [9]         HighTri=127.561
; [10]   HighQuartile=260.186
; [11]        HighSex=653.221
; [12]        HighOct=954.981
; [13]        HighDec=1229.15
; [14]            Max=2.05781e+06 <=== outlier(s)
; snip

;----------------------------------------------------------------------
; Regridding section
;----------------------------------------------------------------------
  interp_method      = "conserve" ; bilinear, neareststod, patch, conserve

  Opt                = True
  Opt@ForceOverwrite = True

  Opt@SrcRegional    = True      ; These two are important! Do not
  Opt@DstRegional    = True      ; set if you have a global lat/lon grid
  
  Opt@DstGridLat     = lat2d
  Opt@DstGridLon     = lon2d

  Opt@InterpMethod   = interp_method

  Opt@Debug          = True
  Opt@PrintTimings   = True

  Opt@SrcFileName    = "EDGAR_SCRIP.nc"
  Opt@DstFileName    = "WRF_SCRIP.nc"
  Opt@WgtFileName    = "EDGAR_to_WRF_" + interp_method + ".nc"

; Uncomment any one of these if the grid file and/or weights file is alreaay generated.
;  Opt@SkipSrcGrid     = True
;  Opt@SkipDstGrid     = True
;  Opt@SkipWgtGen      = True

; Uncomment any one of these to remove source/destination/weight 
;  SCRIP grid files after program termination
;  Opt@RemoveSrcFile   = True               
;  Opt@RemoveDstFile   = True               
;  Opt@RemoveWgtFile   = True               


  var_regrid = ESMF_regrid(var,Opt)   ; Regrid 
;
  printVarSummary(var_regrid)
  printMinMax(var_regrid,0)
  print("==================================================")
end
